﻿using System;
using System.ComponentModel;
using System.Data;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Linq;

using Framework.Data;

namespace Framework.Web.UI.CompositeControls
{
    /// <summary>
    /// GridViewのページャー
    /// </summary>
    [DefaultProperty("TargetGridView")]
    [ToolboxData("<{0}:GridViewPager runat=server></{0}:GridViewPager>")]
    public class GridViewPager : CompositeControl
    {
        /// <summary>
        /// ターゲットのGridViewを指定します（必須）
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string TargetGridView { get; set; }

        /// <summary>
        /// デフォルトのソート式を指定します
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string DefaultSortExpression { get; set; }

        /// <summary>
        /// HeaderCssClass
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string HeaderCssClass { get; set; }

        public string SortExpression 
        {
            get { return (ViewState["SortExpression"] == null) ? DefaultSortExpression : ViewState["SortExpression"].ToString(); }
            private set { ViewState["SortExpression"] = value; }
        }

        public int CurrentPageIndex
        {
            get { return (ViewState["CurrentPageIndex"] == null) ? 0 : (int)ViewState["CurrentPageIndex"]; }
            private set { ViewState["CurrentPageIndex"] = value; }
        }

        public int TotalRecords
        {
            get { return (ViewState["TotalRecords"] == null) ? 0 : (int)ViewState["TotalRecords"]; }
            private set { ViewState["TotalRecords"] = value; }
        }

        public int PageCount
        {
            get { return ((TotalRecords - 1) / _gridView.PageSize + 1); }
        }

        /// <summary>
        /// Sessionに保存されたGridViewのデータソースを返します。
        /// </summary>
        private DataView GridViewSrc
        {
            get
            {
                var ret = _gridView.DataSource as DataView;
                if (ret == null)
                {
                    var key = "{0}.{1}".Fmt(this.ClientID, "GridViewSrc");
                    ret = SessionMgr.Load<DataView>(this.Page, key, null);
                    _gridView.DataSource = ret;
                }
                return ret;
            }
            set
            {
                var key = "{0}.{1}".Fmt(this.ClientID, "GridViewSrc");
                SessionMgr.Save(this.Page, key, value); 
            }
        }

        /// <summary>
        /// GridViewのデータソースリクエストイベント
        /// </summary>
        [Category("Custom")]
        public event GridViewSrcRequestEventHandler GridViewSrcRequest;

        /// <summary>
        /// GridViewのSelectedIndexChangedイベントをラップしたイベント
        /// </summary>
        [Category("Custom")]
        public event GridViewPagerSelectEventHandler SelectedIndexChanged;

        /// <summary>
        /// 編集状態でバインド後に呼ばれ、編集行の状態を変更することができます。
        /// </summary>
        [Category("Custom")]
        public event GridViewPagerEditEventHandler RowEditing;

        /// <summary>
        /// GridViewのRowUpdatingイベントをラップしたイベント
        /// </summary>
        [Category("Custom")]
        public event GridViewPagerUpdateEventHandler RowUpdating;

        /// <summary>
        /// GridViewのRowDeletingイベントをラップしたイベント
        /// </summary>
        [Category("Custom")]
        public event GridViewPagerDeleteEventHandler RowDeleting;

        /// <summary>
        /// GridViewのRowCommandイベントをラップしたイベント
        /// </summary>
        [Category("Custom")]
        public event GridViewPagerCommandEventHandler RowCommand;

        private LLinkButton lkbFirst;
        private LLinkButton lkbPrev;
        private LLinkButton lkbNext;
        private LLinkButton lkbLast;
        private LLabel lblTotalRecords;
        private LLabel lblPageCount;
        private GridView _gridView;

        /// <summary>
        /// CreateChildControls
        /// </summary>
        protected override void CreateChildControls()
        {
            lkbFirst = new LLinkButton();
            lkbFirst.Text = LTextMgr.GetText("Framework.GridViewPager.First", "First<<");
            this.Controls.Add(lkbFirst);

            lkbPrev = new LLinkButton();
            lkbPrev.Text = LTextMgr.GetText("Framework.GridViewPager.Prev", "Prev<");
            lkbPrev.Style.Add("margin-left", "2px");
            this.Controls.Add(lkbPrev);

            lkbNext = new LLinkButton();
            lkbNext.Text = LTextMgr.GetText("Framework.GridViewPager.Next", ">Next");
            lkbNext.Style.Add("margin-left", "2px");
            this.Controls.Add(lkbNext);

            lkbLast = new LLinkButton();
            lkbLast.Text = LTextMgr.GetText("Framework.GridViewPager.Last", ">>Last");
            lkbLast.Style.Add("margin-left", "2px");
            this.Controls.Add(lkbLast);

            lblTotalRecords = new LLabel();
            lblTotalRecords.Style.Add("margin-left", "10px");
            this.Controls.Add(lblTotalRecords);

            lblPageCount = new LLabel();
            lblPageCount.Style.Add("margin-left", "10px");
            this.Controls.Add(lblPageCount);

            if (this.DesignMode)
            {
                base.CreateChildControls();
                return;
            }

            _gridView = this.Parent.FindControl(this.TargetGridView) as GridView;
            if (_gridView == null)
            {
                throw new Exception(string.Format("GridView({0})がみつかりません。", this.TargetGridView));
            }

            //CreateChildControls
            base.CreateChildControls();
        }

        /// <summary>
        /// OnInit
        /// </summary>
        /// <param name="e"></param>
        protected override void OnInit(EventArgs e)
        {
            EnsureChildControls();
            lkbFirst.Click += new EventHandler(lkbFirst_Click);
            lkbPrev.Click += new EventHandler(lkbPrev_Click);
            lkbNext.Click += new EventHandler(lkbNext_Click);
            lkbLast.Click += new EventHandler(lkbLast_Click);

            _gridView.AllowPaging = true;
            _gridView.PagerSettings.Visible = false;
            _gridView.Sorting += new GridViewSortEventHandler(_gridView_Sorting);
            _gridView.RowDataBound += new GridViewRowEventHandler(_gridView_RowDataBound);

            if (SelectedIndexChanged != null)
            {
                _gridView.SelectedIndexChanged += new EventHandler(_gridView_SelectedIndexChanged);
            }

            if (RowCommand != null)
            {
                _gridView.RowCommand += new GridViewCommandEventHandler(_gridView_RowCommand);
            }


            _gridView.RowEditing += new GridViewEditEventHandler(_gridView_RowEditing);
            _gridView.RowCancelingEdit += new GridViewCancelEditEventHandler(_gridView_RowCancelingEdit);

            if (RowUpdating != null)
            {
                _gridView.RowUpdating += new GridViewUpdateEventHandler(_gridView_RowUpdating);
            }

            if (this.RowDeleting != null)
            {
                _gridView.RowDeleting += new GridViewDeleteEventHandler(_gridView_RowDeleting);
            }

            //OnInit
            base.OnInit(e);
        }

        /// <summary>
        /// OnPreRender
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPreRender(EventArgs e)
        {
            //並び順の矢印（▲▼）の表示
            if (_gridView.AllowSorting && !string.IsNullOrEmpty(SortExpression) && _gridView.HeaderRow != null)
            {
                foreach (var str in SortExpression.Split(','))
                {
                    string[] sortExp = str.Trim().Split(new char[]{' '}, StringSplitOptions.RemoveEmptyEntries);
                    for (int i = 0; i < _gridView.Columns.Count; i++)
                    {
                        if (sortExp[0] == _gridView.Columns[i].SortExpression)
                        {
                            var arrow = new Label();
                            arrow.Text = (sortExp.Length < 2 || sortExp[1] == "ASC") ? "&#9650;" : "&#9660;";
                            _gridView.HeaderRow.Cells[i].Controls.Add(arrow);
                            break;
                        }
                    }
                }
            }

            //データソースをSessionに保存
            this.GridViewSrc = (DataView)_gridView.DataSource;

            //OnPreRender
            base.OnPreRender(e);
        }

        /// <summary>
        /// _gridViewのSelectedIndexChanged イベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (_gridView.DataKeyNames == null || _gridView.DataKeyNames.Length == 0)
            {
                throw new Exception("対象のGridViewのDataKeyNamesプロパティに主キーを指定してください。");
            }

            //イベント引数作成
            var se = new GridViewPagerSelectEventArgs();
            se.SelectedIndex = _gridView.SelectedIndex;
            se.DataKey = _gridView.SelectedDataKey;

            //イベント呼び出し
            this.SelectedIndexChanged(sender, se);
        }

        /// <summary>
        /// GridViewのRowDataBoundイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_RowDataBound(object sender, GridViewRowEventArgs e)
        {

            //行データをマッピング
            var rowIdx = e.Row.RowIndex;
            if (rowIdx >= 0)
            {
                e.Row.SetMappingData(new MappingData(this.GridViewSrc[rowIdx].Row));
            }

            //ローカライズ
            e.Row.Localize();
        }

        /// <summary>
        /// _gridViewのRowCommandイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_RowCommand(object sender, GridViewCommandEventArgs e)
        {
            //イベント引数作成
            var ue = new GridViewPagerCommandEventArgs()
            {
                CommandName = e.CommandName,
                CommandArgument = e.CommandArgument,
                CommandSource = e.CommandSource
            };

            var c = e.CommandSource as Control;
            if (c != null)
            {
                ue.GridViewRow = c.FindParentControl<GridViewRow>();
            }

            //イベント呼び出し
            this.RowCommand(this, ue);
        }

        /// <summary>
        /// _gridViewのRowEditingイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_RowEditing(object sender, GridViewEditEventArgs e)
        {
            int rowIdx = e.NewEditIndex;

            //データバインド
            _gridView.EditIndex = rowIdx;
            _gridView.DataSource = this.GridViewSrc;
            _gridView.DataBind();

            if (RowEditing != null)
            {
                //イベント引数作成
                var ee = new GridViewPagerEditEventArgs();
                ee.RowIndex = rowIdx;
                ee.GridViewRow = _gridView.Rows[rowIdx];
                ee.MappingData = new MappingData(this.GridViewSrc[rowIdx].Row);

                //イベント呼び出し
                RowEditing(this, ee);
            }
        }

        /// <summary>
        /// _gridViewのRowCancelingEditイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_RowCancelingEdit(object sender, GridViewCancelEditEventArgs e)
        {
            _gridView.EditIndex = -1;
            _gridView.DataSource = this.GridViewSrc;
            _gridView.DataBind();
        }

        /// <summary>
        /// _gridViewのRowUpdatingイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_RowUpdating(object sender, GridViewUpdateEventArgs e)
        {
            var rowIdx = e.RowIndex;

            //イベント引数作成
            var ue = new GridViewPagerUpdateEventArgs()
            {
                RowIndex = rowIdx,
                DataKey = _gridView.DataKeys.Count > 0 ? _gridView.DataKeys[rowIdx] : null,
                GridViewRow = _gridView.Rows[rowIdx],
                OldValues = this.GridViewSrc[rowIdx],
            };

            //イベント呼び出し
            RowUpdating(this, ue);

            //キャンセル情報を元イベントに引き渡す
            e.Cancel = ue.Cancel;

            if (e.Cancel == false)
            {
                if (ue.NewValues != null)
                {
                    //データを更新後の値に入れ替える
                    _gridView.EditIndex = -1;
                    this.GridViewSrc[rowIdx].Row.ItemArray = ue.NewValues.Row.ItemArray;
                    _gridView.DataSource = this.GridViewSrc;
                    _gridView.DataBind();
                }
            }
        }

        /// <summary>
        /// _gridViewのRowDeletingイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_RowDeleting(object sender, GridViewDeleteEventArgs e)
        {
            //イベント引数作成
            var de = new GridViewPagerDeleteEventArgs();
            de.RowIndex = e.RowIndex;
            if (de.RowIndex >= 0)
            {
                de.RowType = DataControlRowType.DataRow;
                de.DataKey = _gridView.DataKeys.Count > 0 ? _gridView.DataKeys[e.RowIndex] : null;
                de.GridViewRow = _gridView.Rows[e.RowIndex];
            }
            else
            {
                de.RowType = DataControlRowType.Footer;
                de.GridViewRow = _gridView.FooterRow;
            }

            //イベント呼び出し
            RowDeleting(this, de);

            //キャンセル情報を元イベントに引き渡す
            e.Cancel = de.Cancel;

            if (e.Cancel == false)
            {
                //グリッドデータを再検索する
                this.GridViewDataBind(false);
            }
        }

        /// <summary>
        /// ページングイベント：First
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void lkbFirst_Click(object sender, EventArgs e)
        {
            if (CurrentPageIndex > 0)
            {
                CurrentPageIndex = 0;
                this.GridViewDataBind(false);
            }
        }

        /// <summary>
        /// ページングイベント：Prev
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void lkbPrev_Click(object sender, EventArgs e)
        {
            if (CurrentPageIndex > 0)
            {
                CurrentPageIndex--;
                this.GridViewDataBind(false);
            }
        }

        /// <summary>
        /// ページングイベント：Next
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void lkbNext_Click(object sender, EventArgs e)
        {
            if (CurrentPageIndex < PageCount - 1)
            {
                CurrentPageIndex++;
                this.GridViewDataBind(false);
            }
        }

        /// <summary>
        /// ページングイベント：Last
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void lkbLast_Click(object sender, EventArgs e)
        {
            if (CurrentPageIndex < PageCount - 1)
            {
                CurrentPageIndex = PageCount - 1;
                this.GridViewDataBind(false);
            }
        }

        /// <summary>
        /// _gridViewのSortingイベント
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void _gridView_Sorting(object sender, GridViewSortEventArgs e)
        {
            string sortOrder = "ASC";
            if (!string.IsNullOrEmpty(SortExpression))
            {
                string[] curSort = SortExpression.Split(' ');
                if (curSort[0] == e.SortExpression && (curSort.Length < 2 || curSort[1] == "ASC"))
                {
                    sortOrder = "DESC";
                }
            }
            SortExpression = string.Format("{0} {1}", e.SortExpression, sortOrder);
            CurrentPageIndex = 0;
            this.GridViewDataBind(false);
        }

        /// <summary>
        /// データソースをリクエストし、戻ってきたデータをGridViewにバインドします。
        /// </summary>
        /// <param name="initPaging">ページ・ソートを初期状態にするか？</param>
        public void GridViewDataBind(bool initPaging)
        {
            if (this.GridViewSrcRequest == null)
            {
                throw new Exception("GridViewSrcRequestイベントを登録してください。");
            }

            if (initPaging)
            {
                //表示ページを初期化する
                this.CurrentPageIndex = 0;
                this.SortExpression = this.DefaultSortExpression;
            }

            while (true)
            {
                //呼び出し元に表示ページのデータソースをリクエストする
                var e = new GridViewSrcRequestEventArgs();
                e.MaxRecords = _gridView.PageSize;
                e.SortExpression = this.SortExpression;
                e.StartRecord = this.CurrentPageIndex * _gridView.PageSize + 1;
                this.GridViewSrcRequest(this, e);

                if ((this.CurrentPageIndex > 0) && (e.DataSource == null || e.DataSource.Count == 0))
                {
                    this.CurrentPageIndex--;
                }
                else
                {
                    _gridView.DataSource = e.DataSource;
                    _gridView.DataBind();

                    this.TotalRecords = e.TotalRecords;
                    this.SetPager();

                    if (e.DataSource == null || e.DataSource.Count == 0)
                    {
                        //フッターのローカライズ
                        if (_gridView.ShowFooter == true)
                        {
                            _gridView.FooterRow.Localize();
                        }
                    }
                    break;
                }
            }

        }

        /// <summary>
        /// ページャーの表示設定
        /// </summary>
        private void SetPager()
        {
            lkbFirst.Enabled = (CurrentPageIndex > 0);
            lkbPrev.Enabled = (CurrentPageIndex > 0);
            lkbNext.Enabled = (CurrentPageIndex < PageCount - 1);
            lkbLast.Enabled = (CurrentPageIndex < PageCount - 1);
            lblTotalRecords.Text = string.Format("{1}: {0}", TotalRecords, LTextMgr.GetText("Framework.GridViewPager.Total", "Total Records"));
            lblPageCount.Text = string.Format("{2} {0}/{1}", (CurrentPageIndex + 1), PageCount, LTextMgr.GetText("Framework.GridViewPager.Page", "Page"));
        }
    }
}
